﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using Newtonsoft.Json;
using Ext.Net;
using System.IO;
using System.Data;
using DataAccess;
using System.Collections;
using System.Web.Script.Serialization;

[DirectMethodProxyID(IDMode = DirectMethodProxyIDMode.Alias, Alias = "UCADMIN")]

public partial class ucAdministration : System.Web.UI.UserControl
{
    //base master
    public BaseMaster BaseMstr { get; set; }

    /// <summary>
    /// edit mode
    /// </summary>
    public string EditMode
    {
        get
        {
            string strValue = "";
            if (Session["ucADMINEditMode"] != null)
            {
                strValue = Session["ucADMINEditMode"].ToString();
            }

            return strValue;
        }
        set { Session["ucADMINEditMode"] = Convert.ToString(value); }
    }

    /// <summary>
    /// treatment id
    /// </summary>
    public string TreatmentID
    {
        get
        {
            string strValue = "";
            if (Session["ucADMINTreatmentID"] != null)
            {
                strValue = Session["ucADMINTreatmentID"].ToString();
            }

            return strValue;
        }
        set { Session["ucADMINTreatmentID"] = Convert.ToString(value); }
    }

    /// <summary>
    /// program id
    /// </summary>
    public string ProgramID
    {
        get
        {
            string strValue = "";
            if (Session["ucADMINProgramID"] != null)
            {
                strValue = Session["ucADMINProgramID"].ToString();
            }

            return strValue;
        }
        set { Session["ucADMINProgramID"] = Convert.ToString(value); }
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// show new treatment popup
    /// </summary>
    public void OnNewTreatment()
    {
        EditMode = "INSERT";
        winAdminTreatment.Title = "Add Episode";

        //load the controls on the window
        dfDlgTreatmentAdminDate.Clear();        
        dfDlgTreatmentDischargeDate.Clear();

        //1 = outpatient, 2=inpatient
        rdoDlgTreatmentInPatient.Checked = true;
        rdoDlgTreatmentOutPatient.Checked = false;

        tfDlgTreatmentTitle.Text = "";

        //clear warnings
        dfDlgTreatmentAdminDate.ClearInvalid();
        dfDlgTreatmentDischargeDate.ClearInvalid();
        btnEpisodeSave.Disabled = false;

        //show the window
        winAdminTreatment.Show();
        tfDlgTreatmentTitle.Focus(false, 100);
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// show new program popup
    /// </summary>
    public void OnNewProgram()
    {
        EditMode = "INSERT";
        winAdminProgram.Title = "Add Program";

        //clear the window controls
        dfDlgProgramAdminDate.Clear();
        dfDlgProgramDischargeDate.Clear();
        cboDlgProgramCPA.SelectedItems.Clear();
        cboDlgProgramCPA.UpdateSelectedItems();

        //clear warnings
        dfDlgProgramAdminDate.ClearInvalid();
        dfDlgProgramDischargeDate.ClearInvalid();
        btnTreatSave.Disabled = false;

        //show the window
        cboDlgProgramCPA.UpdateLayout();

        winAdminProgram.Show();
        dfDlgProgramAdminDate.Focus(false, 100);
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// show edit treatment popup
    /// </summary>
    public void OnEditTreatment(string strJsonRecord)
    {
        EditMode = "UPDATE";
        winAdminTreatment.Title = "Update Episode";

        //get the treament id form the record
        long lTreatmentID = CDataUtils2.ToLong(CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord));
        
        TreatmentID = CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord);

        //get the record
        CPatTreatment treat = new CPatTreatment();
        DataSet ds = treat.GetTreatmentDS(BaseMstr, BaseMstr.SelectedPatientID, lTreatmentID);

        //load the controls on the window
        dfDlgTreatmentAdminDate.SelectedDate = CDataUtils2.GetDSDateTimeValue(ds, "INITIAL_DATE");

        dfDlgTreatmentDischargeDate.Clear();
        DateTime dt = CDataUtils2.GetDSDateTimeValue(ds, "END_DATE");
        if (!CDataUtils2.IsDateNull(dt))
        {
            dfDlgTreatmentDischargeDate.SelectedDate = dt;
        }

        //1 = outpatient, 2=inpatient
        long lType = CDataUtils2.GetDSLongValue(ds, "TREATMENT_TYPE_ID");
        rdoDlgTreatmentInPatient.Checked = false;
        rdoDlgTreatmentOutPatient.Checked = false;
        if (lType == 1)
        {
            rdoDlgTreatmentOutPatient.Checked = true;
        }
        if (lType == 2)
        {
            rdoDlgTreatmentInPatient.Checked = true;
        }

        tfDlgTreatmentTitle.Text = CDataUtils2.GetDSStringValue(ds, "TREATMENT_TITLE");

        //show the window
        winAdminTreatment.Show();
        tfDlgTreatmentTitle.Focus(false, 100);
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// delete treatment
    /// </summary>
    public void OnDeleteTreatment(string strJsonRecord)
    {
        //get the treament id form the record
        long lTreatmentID = CDataUtils2.ToLong(CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord));
        
        TreatmentID = CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord);

        CPatTreatment treat = new CPatTreatment();
        treat.DeleteTreatment(BaseMstr, BaseMstr.SelectedPatientID, lTreatmentID);

        //clear programs, load treatments
        storTreatmentPrograms.SetDataFromJson(CDataUtils2.GetEmptyJSON());
        this.LoadTreatments();
            
    }


    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// show edit program popup
    /// </summary>
    public void OnEditProgram(string strJsonRecord)
    {
        EditMode = "UPDATE";
        winAdminProgram.Title = "Update Program";

        //get the treament id form the record
        long lTreatmentID = CDataUtils2.ToLong(CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord));
        long lProgramID = CDataUtils2.ToLong(CDataUtils2.GetJsonRecordValue("PROGRAM_ID", strJsonRecord));

        TreatmentID = CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord);
        ProgramID = CDataUtils2.GetJsonRecordValue("PROGRAM_ID", strJsonRecord);

        //get the record
        CPatTreatment treat = new CPatTreatment();
        DataSet ds = treat.GetTreatmentProgramDS(BaseMstr, BaseMstr.SelectedPatientID, lTreatmentID, lProgramID);

        //load the controls on the window
        dfDlgProgramAdminDate.SelectedDate = CDataUtils2.GetDSDateTimeValue(ds, "INITIAL_DATE");

        dfDlgProgramDischargeDate.Clear();
        DateTime dt = CDataUtils2.GetDSDateTimeValue(ds, "END_DATE");
        if (!CDataUtils2.IsDateNull(dt))
        {
            dfDlgProgramDischargeDate.SelectedDate = dt;
        }

        cboDlgProgramCPA.SelectedItems.Clear();

        Ext.Net.ListItem li = new Ext.Net.ListItem();
        li.Value = Convert.ToString(CDataUtils2.GetDSLongValue(ds, "CPA_ID"));
        li.Mode = ParameterMode.Raw;
        cboDlgProgramCPA.SelectedItems.Add(li);
        cboDlgProgramCPA.UpdateSelectedItems();
     
        winAdminProgram.Show();
        dfDlgProgramAdminDate.Focus(false, 100);
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// delete a program
    /// </summary>
    public void OnDeleteProgram(string strJsonRecord)
    {
        //get the treament id form the record
        long lTreatmentID = CDataUtils2.ToLong(CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord));
        long lprogramID = CDataUtils2.ToLong(CDataUtils2.GetJsonRecordValue("PROGRAM_ID", strJsonRecord));

        TreatmentID = CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord);
        ProgramID = CDataUtils2.GetJsonRecordValue("PROGRAM_ID", strJsonRecord);

        //delete the program
        CPatTreatment treat = new CPatTreatment();
        treat.DeleteTreatmentProgram(BaseMstr, BaseMstr.SelectedPatientID, lprogramID);

       //load the progrms
        this.LoadTreatmentProgramList(lTreatmentID);
    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// on episode treatment select 
    /// </summary>
    public void OnTreatmentSelect(string strJsonRecord)
    {
        //get the treament id form the record
        long lTreatmentID = CDataUtils2.ToLong(CDataUtils2.GetJsonRecordValue("TREATMENT_ID", strJsonRecord));

        //cache the treatment id
        TreatmentID = Convert.ToString(lTreatmentID);
        
        //load the programs for this treatment
        LoadTreatmentProgramList(lTreatmentID);

        //enable the new program button
        btnNewProgram.Enable();

        //enable treatment edit options
        btnUCADMNTreatEdit.Enable();
        btnUCADMNTreatDel.Enable();

        //disable program edit options
        btnUCADMNProgEdit.Disable();
        btnUCADMNProgDel.Disable();
    }


    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// on episode treatment select 
    /// </summary>
    public void OnProgramSelect(string strJsonRecord)
    {   
        //enable program edit options
        btnUCADMNProgEdit.Enable();
        btnUCADMNProgDel.Enable();
    }

    /// <summary>
    /// validate the treatment dates
    /// </summary>
    /// <param name="dtStart"></param>
    /// <param name="dtEnd"></param>
    /// <returns></returns>
    protected bool ValidateTreatmentDates(DateTime dtStart, DateTime dtEnd)
    {
        //must have admission/start date
        if (dtStart.Year == 1 || CDataUtils2.IsDateNull(dtStart))
        {
            BaseMstr.ShowSystemFeedback("Please select a valid initial date!", "Error");
            return false;
        }

        //end date (if entered) must come after inital date
        if (dtEnd.Year != 1)
        {
            if (!CDataUtils2.IsDateNull(dtEnd))
            {
                if (dtStart > dtEnd)
                {
                    BaseMstr.ShowSystemFeedback("Episode end date must be greater than or equal to initial date!", "Error");
                    return false;
                }
            }
        }

        return true;
    }

    /// <summary>
    /// validate a programs dates
    /// </summary>
    /// <param name="dtStart"></param>
    /// <param name="dtEnd"></param>
    /// <returns></returns>
    protected bool ValidateProgramDates(DateTime dtStart, DateTime dtEnd)
    {
        //must have admission/start date
        if (dtStart.Year == 1 || CDataUtils2.IsDateNull(dtStart))
        {
            BaseMstr.ShowSystemFeedback("Please select a valid initial date!", "Error");
            return false;
        }

        //end date (if entered) must come after inital date
        if (dtEnd.Year != 1)
        {
            if (!CDataUtils2.IsDateNull(dtEnd))
            {
                if (dtStart > dtEnd)
                {
                    BaseMstr.ShowSystemFeedback("Program end date must be greater than or equal to initial date!", "Error");
                    return false;
                }
            }
        }

        //get the treatment record for the selected treatment
        long lTreatmentID = CDataUtils2.ToLong(TreatmentID);

        CPatTreatment treat = new CPatTreatment();
        DataSet dst = treat.GetTreatmentDS(BaseMstr,
                                           BaseMstr.SelectedPatientID,
                                           lTreatmentID);

        DateTime dtProgStart = CDataUtils2.GetDSDateTimeValue(dst, "INITIAL_DATE");
        DateTime dtProgEnd = CDataUtils2.GetDSDateTimeValue(dst, "END_DATE");

     
        //start of the program must be >= the treatment start
        if (dtStart < dtProgStart)
        {
            BaseMstr.ShowSystemFeedback("Program initial date cannot be less than episode initial date!", "Error");
            return false;
        }

        //treatment has discharge date
        if (!CDataUtils2.IsDateNull(dtProgEnd))
        {
            //program has discharge
            if (dtEnd.Year != 1)
            {
                if (!CDataUtils2.IsDateNull(dtEnd))
                {
                    //end of program must be <= treatment discharge
                    if (dtEnd > dtProgEnd)
                    {
                        BaseMstr.ShowSystemFeedback("Program end date must be less than or equal to episode end date!", "Error");
                        return false;
                    }

                    //end of program must be >= treatment start
                    if (dtEnd < dtProgStart)
                    {
                        BaseMstr.ShowSystemFeedback("Program end date must be greater than or equal to episode initial date!", "Error");
                        return false;
                    }
                }
            }
        }              

        return true;
    }

   
    /// <summary>
    /// load treatment programs grid
    /// </summary>
    /// <param name="lTreatmentID"></param>
    protected void LoadTreatmentProgramList(long lTreatmentID)
    {

        //clear grid panle store
        storTreatmentPrograms.SetDataFromJson(CDataUtils2.GetEmptyJSON());


        CPatTreatment treat = new CPatTreatment();
        DataSet ds = treat.GetTreatmentProgramDS(BaseMstr, BaseMstr.SelectedPatientID, lTreatmentID);

        if (!CDataUtils2.IsEmpty(ds))
        {
            CDataConverter cdc = new CDataConverter();
            string strJSON = cdc.GetJsonDSString(ds);

            storTreatmentPrograms.SetDataFromJson(strJSON);

        }
    }

    /// <summary>
    /// Load the treatments for the patient
    /// </summary>
    protected void LoadTreatments()
    {
        //clear the treatment store
        storEpisode.SetDataFromJson(CDataUtils2.GetEmptyJSON());

        //load treatment store
        CPatTreatment treat = new CPatTreatment();
        DataSet dsTreat = treat.GetTreatmentDS(BaseMstr, BaseMstr.SelectedPatientID);
        if (!CDataUtils2.IsEmpty(dsTreat))
        {
            CDataConverter cdc = new CDataConverter();
            string strJSON = cdc.GetJsonDSString(dsTreat);

            storEpisode.SetDataFromJson(strJSON);
        }
  }

    /// <summary>
    /// load cpa drop down list
    /// </summary>
    protected void LoadCPA()
    {
        //clear the CPA store
        storProgramCPA.SetDataFromJson(CDataUtils2.GetEmptyJSON());

        //load cpa store
        CCPA cpa = new CCPA();
        DataSet dsCPA = cpa.GetCPADS(BaseMstr, BaseMstr.RegionID, BaseMstr.SiteID);
        DataSet dsView = new DataSet();

        dsView = dsCPA.Clone();

        for (int i = 0; i < dsCPA.Tables[0].Rows.Count; i++)
        {
            if (dsCPA.Tables[0].Rows[i].ItemArray[1].ToString() != "Open Pathway")
            {
                dsView.Tables[0].ImportRow(dsCPA.Tables[0].Rows[i]);
            }
        }

        if (!CDataUtils2.IsEmpty(dsView))
        {
            CDataConverter cdc = new CDataConverter();
            string strJSON = cdc.GetJsonDSString(dsView);

            storProgramCPA.SetDataFromJson(strJSON);
        }

        //clear selections
        cboDlgProgramCPA.SelectedItems.Clear();
        cboDlgProgramCPA.UpdateSelectedItems();

        cboDlgProgramCPA.UpdateLayout();
    }
   


    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// insert/update a treatment
    /// </summary>
    protected void OnSaveTreatment(object sender, DirectEventArgs e)
    {
        if (EditMode == "INSERT")
        {
            InsertTreatment();
        }
        if (EditMode == "UPDATE")
        {
            UpdateTreatment();
        }
    }

    /// <summary>
    /// update a treatment
    /// </summary>
    protected void UpdateTreatment()
    {
        //todo validate 

        long lTreatmentID = CDataUtils2.ToLong(TreatmentID);

        //must have admission date
        DateTime dtAdmission = dfDlgTreatmentAdminDate.SelectedDate;
        if (dtAdmission.Year == 1)
        {
            BaseMstr.ShowSystemFeedback("Please select a valid initial date!", "Error");
            return;
        }

        //must have title
        if (tfDlgTreatmentTitle.Text.Length < 1)
        {
            BaseMstr.ShowSystemFeedback("Please enter a valid title!", "Error");
            return;
        }

        //check the date range, method will show error
        if (!ValidateTreatmentDates(dtAdmission, dfDlgTreatmentDischargeDate.SelectedDate))
        {
            return;
        }


        //adjust discharge to null if needed
        DateTime dtDischarge = dfDlgTreatmentDischargeDate.SelectedDate;
        if (dtAdmission.Year == 1)
        {
            dtDischarge = CDataUtils2.GetNullDate();
        }

        //treatment type id (inpatient outpatient)
        long lTreatmentTypeID = 2; //2=inpatient
        if (rdoDlgTreatmentOutPatient.Checked)
        {
            lTreatmentTypeID = 1; //1=outpatient
        }

        CPatTreatment treat = new CPatTreatment();

        //cannot add a second treatment of in or out if already an in or out if there is currently an open one
        DataSet dsTreat = treat.GetTreatmentDS(BaseMstr,
                                              BaseMstr.SelectedPatientID);

        foreach (DataTable dt in dsTreat.Tables)
        {
            foreach (DataRow dr in dt.Rows)
            {
                if (CDataUtils2.GetDSLongValue(dr, "TREATMENT_ID") != lTreatmentID)
                {
                    //types match
                    long lType = CDataUtils2.GetDSLongValue(dr, "TREATMENT_TYPE_ID");
                    if (lType == lTreatmentTypeID)
                    {
                        //its open
                        if (CDataUtils2.IsDateNull(CDataUtils2.GetDSDateTimeValue(dr, "END_DATE")))
                        {
                            BaseMstr.ShowSystemFeedback("There is already an open " + CDataUtils2.GetDSStringValue(dr, "TREATMENT_TYPE") + " for this patient.", "Error");
                            return;
                        }
                    }
                }
            }
        }

        //program dates cannot exceed new episode dates
        DataSet dsProg = treat.GetTreatmentProgramDS(BaseMstr, BaseMstr.SelectedPatientID, lTreatmentID);
        if (!CDataUtils2.IsEmpty(dsProg))
        {
            foreach(DataTable dt in dsProg.Tables)
            {
                foreach (DataRow dr in dt.Rows)
                {
                    if (CDataUtils2.GetDSDateTimeValue(dr, "INITIAL_DATE") < dtAdmission)
                    {
                        BaseMstr.ShowSystemFeedback("This Episode has a Program that starts before the chosen Initial Date.~#(br/)#~Please review the Program dates before making this change.", "Error");
                        return;
                    }

                    if (!CDataUtils2.IsDateNull(dtDischarge))
                    {
                        if (CDataUtils2.GetDSDateTimeValue(dr, "INITIAL_DATE") > dtDischarge)
                        {
                            BaseMstr.ShowSystemFeedback("This Episode has a Program that starts after the chosen End Date.~#(br/)#~Please review the Program dates before making this change.", "Error");
                            return;
                        }
                    
                        if (!CDataUtils2.IsDateNull(CDataUtils2.GetDSDateTimeValue(dr, "END_DATE")))
                        {
                            if (CDataUtils2.GetDSDateTimeValue(dr, "END_DATE") > dtDischarge)
                            {
                                BaseMstr.ShowSystemFeedback("This Episode has a Program that ends after the chosen End Date.~#(br/)#~Please review the Program dates before making this change.", "Error");
                                return;
                            }
                        }
                    }
                }
            }
        }

        //update treatment
        treat.UpdateTreatment( BaseMstr, 
                               BaseMstr.SelectedPatientID,
                               lTreatmentID,
                               lTreatmentTypeID,
                               tfDlgTreatmentTitle.Text,
                               dtAdmission,
                               dtDischarge);

        //reload the treatments
        LoadTreatments();

        //select the treatment we jsut updated
        gpTreatments.GetSelectionModel().Select(lTreatmentID);

        winAdminTreatment.Close();
        
    }

    /// <summary>
    /// update a program
    /// </summary>
    protected void UpdateProgram()
    {
        //todo validate

        long lTreatmentID = CDataUtils2.ToLong(TreatmentID);
        long lProgramID = CDataUtils2.ToLong(ProgramID);

        //must have admission date
        DateTime dtAdmission = dfDlgProgramAdminDate.SelectedDate;
        if (dtAdmission.Year == 1)
        {
            BaseMstr.ShowSystemFeedback("Please select a valid initial date!", "Error");
            return;
        }

        //must have cpa picked
        if (cboDlgProgramCPA.SelectedItem == null)
        {
            BaseMstr.ShowSystemFeedback("Please select a valid clinical program!", "Error");
            return;

        }

        //validate program dates. method will show error if failed
        if (!ValidateProgramDates(dtAdmission, dfDlgProgramDischargeDate.SelectedDate))
        {
            return;
        }


        //adjust discharge to null if needed
        DateTime dtDischarge = dfDlgProgramDischargeDate.SelectedDate;
        if (dtAdmission.Year == 1)
        {
            dtDischarge = CDataUtils2.GetNullDate();
        }

        long lCPAID = CDataUtils2.ToLong(cboDlgProgramCPA.SelectedItem.Value);
        CPatTreatment treat = new CPatTreatment();
        treat.UpdateTreatmentProgram(BaseMstr,
                                     BaseMstr.SelectedPatientID,
                                     lTreatmentID,
                                     lProgramID,
                                     lCPAID,
                                     dtAdmission,
                                     dtDischarge);

        //reload the treatments
        LoadTreatmentProgramList(lTreatmentID);

        //select the program we jsut updated
        gpTreatmentPrograms.GetSelectionModel().Select(lProgramID);

        //close the edit window
        winAdminProgram.Close();

    }


    /// <summary>
    /// add a treatment
    /// </summary>
    protected void InsertTreatment()
    {
        //check form

        //must have admission date
        DateTime dtAdmission = dfDlgTreatmentAdminDate.SelectedDate;
        if (dtAdmission.Year == 1)
        {
            BaseMstr.ShowSystemFeedback("Please select a valid initial date!", "Error");
            return;
        }
        
        //must have title
        if (tfDlgTreatmentTitle.Text.Length < 1)
        {
            BaseMstr.ShowSystemFeedback("Please enter a valid title!", "Error");
            return;
        }

        //check the date range, method will show error
        if (!ValidateTreatmentDates(dtAdmission, dfDlgTreatmentDischargeDate.SelectedDate))
        {
            return;
        }
        
        CPatTreatment treat = new CPatTreatment();

        //adjust discharge to null if needed
        DateTime dtDischarge = dfDlgTreatmentDischargeDate.SelectedDate;
        if (dtAdmission.Year == 1)
        {
            dtDischarge = CDataUtils2.GetNullDate();
        }

        //treatment type id (inpatient outpatient)
        long lTreatmentTypeID = 2; //2=inpatient
        if (rdoDlgTreatmentOutPatient.Checked)
        {
            lTreatmentTypeID = 1; //1=outpatient
        }
        
        //cannot add a second treatment of in or out if already an in or out if there is currently an open one
        DataSet dsprog = treat.GetTreatmentDS(BaseMstr,
                                              BaseMstr.SelectedPatientID);

        foreach (DataTable dt in dsprog.Tables)
        {
            foreach (DataRow dr in dt.Rows)
            {
                //types match
                long lType = CDataUtils2.GetDSLongValue(dr, "TREATMENT_TYPE_ID");
                if(lType == lTreatmentTypeID)
                {
                    //its open
                    if (CDataUtils2.IsDateNull(CDataUtils2.GetDSDateTimeValue(dr, "END_DATE")))
                    {
                        BaseMstr.ShowSystemFeedback("There is already an open " + CDataUtils2.GetDSStringValue(dr, "TREATMENT_TYPE") + " for this patient.", "Error");
                        return;
                    }
                }
            }
        }
 

        //insert the treatment into the database
        long lTreatmentID = 0;
        if (treat.InsertTreatment(BaseMstr,
                                  BaseMstr.SelectedPatientID,
                                  lTreatmentTypeID,
                                  dfDlgTreatmentAdminDate.SelectedDate,
                                  dtDischarge,
                                  tfDlgTreatmentTitle.Text,
                                  out lTreatmentID))
        {

            //reload combo
            LoadTreatments();

            //clear the program grid panel
            storTreatmentPrograms.SetDataFromJson(CDataUtils2.GetEmptyJSON());

            //load the programs for this treatment
            LoadTreatmentProgramList(lTreatmentID);

            //select the treatment in the list
            gpTreatments.GetSelectionModel().Select(lTreatmentID);

            //enable the new program button
            btnNewProgram.Enable();

            //close the window
            winAdminTreatment.Close();
        }

    }

    [DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// 
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void OnSaveProgram(object sender, DirectEventArgs e)
    {
        if (EditMode == "INSERT")
        {
            InsertProgram();
        }
        if (EditMode == "UPDATE")
        {
            UpdateProgram();
        }

    }
      
    /// <summary>
    /// insert a program
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void InsertProgram()
    {
        //check form

        //must have admission date
        DateTime dtAdmission = dfDlgProgramAdminDate.SelectedDate;
        if (dtAdmission.Year == 1)
        {
            BaseMstr.ShowSystemFeedback("Please select a valid initial date!", "Error");
            return;
        }

        //must have cpa picked
        if (cboDlgProgramCPA.SelectedItem == null)
        {
            BaseMstr.ShowSystemFeedback("Please select a valid clinical program!", "Error");
            return;

        }

        //validate program dates. method will show error if failed
        if (!ValidateProgramDates(dtAdmission, dfDlgProgramDischargeDate.SelectedDate))
        {
            return;
        }

        
        //adjust discharge to null if needed
        DateTime dtDischarge = dfDlgProgramDischargeDate.SelectedDate;
        if (dtAdmission.Year == 1)
        {
            dtDischarge = CDataUtils2.GetNullDate();
        }

        //insert a new treatment program
        long lProgramID = 0;
        long lTreatmentID = CDataUtils2.ToLong(TreatmentID);
        long lCPAID = CDataUtils2.ToLong(cboDlgProgramCPA.SelectedItem.Value);

        //cannot add a second program if there is currently an open one
        CPatTreatment treat = new CPatTreatment();
        DataSet dsprog = treat.GetTreatmentProgramDS(BaseMstr,
                                                     BaseMstr.SelectedPatientID,
                                                     lTreatmentID);
        foreach (DataTable dt in dsprog.Tables)
        {
            foreach (DataRow dr in dt.Rows)
            {
         
                if(CDataUtils2.GetDSLongValue(dr, "CPA_ID") == lCPAID)
                {
                    if(CDataUtils2.IsDateNull(CDataUtils2.GetDSDateTimeValue(dr, "END_DATE")))
                    {
                        BaseMstr.ShowSystemFeedback("There is already an open program for " + cboDlgProgramCPA.SelectedItem.Text, "Error");
                        return;
                    }
                }
            }
        }
 
     
        if (treat.InsertTreatmentProgram(BaseMstr,
                                          BaseMstr.SelectedPatientID,
                                          lTreatmentID,
                                          lCPAID,
                                          dfDlgProgramAdminDate.SelectedDate,
                                          dtDischarge,
                                          out lProgramID))
        {

            //reload the grid
            LoadTreatmentProgramList(lTreatmentID);

            //select the program
            gpTreatmentPrograms.GetSelectionModel().Select(lProgramID);

            //clear elements for next save
            dfDlgTreatmentAdminDate.Clear();
            dfDlgTreatmentDischargeDate.Clear();
            tfDlgTreatmentTitle.Text = "";
            rdoDlgTreatmentInPatient.Checked = true;
            
            //close the window
            winAdminProgram.Close();
        }
                
    }


    /// <summary>
    /// page load
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e)
    {
        if (!IsPostBack)
        {
            LoadCPA();
            LoadTreatments();
            EditMode = "INSERT";

            //disable buttons until something selected
            btnUCADMNTreatEdit.Disable();
            btnUCADMNTreatDel.Disable();
            btnUCADMNProgEdit.Disable();
            btnUCADMNProgDel.Disable();
        }
    }
}